/**
* \file: automounter.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <stdio.h>
#include <signal.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <limits.h>
#include <string.h>
#include <execinfo.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <assert.h>

#include "utils/logger.h"
#include "utils/version_info.h"
#include "utils/global_constants.h"
#include "model/device.h"
#include "model/device_list.h"
#include "control/device_fsm.h"
#include "control/automounter.h"
#include "control/mount.h"
#include "control/blacklist.h"
#include "control/configuration.h"
#include "control/watch.h"
#include "control/signal_handler.h"
#include "app_iface/app_iface_intern.h"

#include "model/device_list.h"

#include "backends/device_handler.h"

//this value is used for callstack generation in automounter_die_on_resource_issues(). It defines the maximum depth
//the callstack could ever have
#define MAX_CALLSTACK_DEPTH 100

typedef enum {
	_UNSET,
	RUNNING,
	SHUTDOWN_REQUEST,
	SHUTTING_DOWN,
	DONE
} automounter_mainloop_state_t;

static automounter_mainloop_state_t automounter_mainloop_state=_UNSET;

static void automounter_init_logging(int argc, char *argv[]);
static void automounter_early_parse_args(int argc,char *argv[],
		const char **proc_name, logger_loglevel_t *loglevel, bool *console_enabled);
static void automounter_deinit_device_handler(void);

static error_code_t automounter_create_run_dir(void);
static void automounter_remove_run_dir(void);

static bool automounter_unmountall_request(void);
static void automounter_evaluate_mainloop_state_machine(void);

static error_code_t automounter_blacklist_mounted_partitions(void);

static error_code_t automounter_check_mountdir(void);


//---------------------------------------- API members ---------------------------------------------
error_code_t automounter_init(int argc, char *argv[])
{
	error_code_t result;

	//initialize this before the signals are redirected
	automounter_mainloop_state=RUNNING;

	automounter_init_logging(argc,argv);

	result=watch_init();

	if (result==RESULT_OK)
		result=signal_handler_init();

	if (result==RESULT_OK)
		result=device_list_init();

	if (result==RESULT_OK)
		result=blacklist_init();

	if (result==RESULT_OK)
		result=mount_init();

	if (result!=RESULT_OK)
		return result;

	//start now reading in the configuration. The configuration module sets up directly
	//some of the modules already initialized while reading in the configuration files and parsing
	//the arguments

	//parses the arguments
	result=configuration_init(argc,argv);

	//reads out the configuration file (either the default one or the one given as command line option)
	if (result==RESULT_OK)
		result=configuration_parse_configuration_file();

	//set loglevel again, command line options or the configuration file might have changed the initial value
	logger_set_log_level(configuration_get_loglevel());
	logger_set_console_enabled(configuration_is_console_enabled());

	//now we are starting modules that are using or might use configuration parameters for their initialization

	//initialize modules without any dependency to the configuration
	if (result==RESULT_OK)
		result=automounter_create_run_dir();

	if (result==RESULT_OK)
		result=automounter_check_mountdir();

	if (result==RESULT_OK)
		result=app_iface_init();

	if (result==RESULT_OK)
		result=device_handler_init_all();

	//now we are looking up mounted partitions and blacklist them and the underlying devices
	if (result==RESULT_OK)
		result=automounter_blacklist_mounted_partitions();

	return result;
}

bool automounter_shall_daemonize(void)
{
	return configuration_shall_daemonize();
}

error_code_t automounter_mainloop(void)
{
	watch_process_event_result_t result;

	while(automounter_mainloop_state!=DONE)
	{
		result=watch_wait_for_and_process_event();
		if (result==WATCH_EVENT_PROCESSED || result==WATCH_INTERRUPTED)
			automounter_evaluate_mainloop_state_machine();

		if (result==WATCH_ERROR)
			return RESULT_NORESOURCE;
	}

	return RESULT_OK;
}

void automounter_deinit(void)
{
	logger_log_info("Automounter exiting");
	automounter_deinit_device_handler();
	app_iface_deinit();
	mount_deinit();
	signal_handler_deinit();
	watch_deinit();
	blacklist_deinit();
	device_list_deinit();
	automounter_remove_run_dir();
	logger_deinit();
	configuration_deinit();
}

void automounter_term(void)
{
        if (automounter_mainloop_state == RUNNING)
        {
                //first kill: we are shutting down normally unmounting all we have under control
	        automounter_mainloop_state = SHUTDOWN_REQUEST;
        }
	else
	{
		//second kill: it seems we are too slow, lets go down immediately
		automounter_mainloop_state = DONE;
	}
}

void automounter_die_on_resource_issues(void)
{
	void *buffer[MAX_CALLSTACK_DEPTH];
	int callstack_size;
	int kmsg_bug_fd;

	//escalation strategy:
	//	- when we have called here, we are already sure that there is no way out but exiting
	//	- we are now trying to create a callstack and put it into the kernel message buffer log
	//	- we then plan to exit with ENOMEM so that systemd can principally restart us
	//	- if the callstack generation fails, we are raising an assert so that exception-handler can create the
	//		callstack for us. Depending on the configuration, this will finally lead to a reset of the target!!!
	logger_log_error("Automounter is faced some resources problems. Probably we ran "
			"out of memory. Best solution is now to die bravely.");

	logger_log_error("Started a call stack generation.");
	callstack_size=backtrace(buffer,MAX_CALLSTACK_DEPTH);
	logger_log_error("Got a callstack size of depth: %d.",callstack_size);
	logger_log_error("Trying now to resolve symbols and dumping the callstack into kernel message buffer.");

	kmsg_bug_fd=open("/dev/kmsg",O_WRONLY);
	if (kmsg_bug_fd==-1)
	{
		logger_log_error("Error opening kernel message buffer.");
		logger_log_error("Callstack generation finally failed.");
		logger_log_error("We are now raising an assert to let the exception handler create a callstack for us.");
		assert(false);
	}
	else
	{
		backtrace_symbols_fd(buffer, callstack_size, kmsg_bug_fd);
		close(kmsg_bug_fd);
	}

	exit(ENOMEM);
}
//-------------------------------------------------------------------------------------------------

//------------------------------------- private members -------------------------------------------

static void automounter_early_parse_args(int argc,char *argv[],
		const char **proc_name, logger_loglevel_t *loglevel, bool *console_enabled)
{
	int i;
	*proc_name=argv[0];

	for (i=1;i<argc;i++)
	{
		if (strcmp(argv[i],"-q")==0 || strcmp(argv[i],"--quiet")==0)
			*console_enabled=false;
		if (strcmp(argv[i],"-l")==0 || strcmp(argv[i],"--loglevel")==0)
		{
			if (i<argc-1)
				logger_parse_loglevel(argv[i+1],loglevel);
		}
		if (strstr(argv[i],"--loglevel=")!=NULL)
		{
			const char *value;
			value=strchr(argv[i],'=');
			//to satisfy lin(t),sorry
			if (value!=NULL)
			{
				if (value[1]!='\0')
					logger_parse_loglevel(value,loglevel);
			}
		}
	}
}

static void automounter_init_logging(int argc, char *argv[])
{
	const char *proc_name;
	logger_loglevel_t loglevel;
	bool console_enabled;

	//get default values
	loglevel=configuration_get_loglevel();
	console_enabled=configuration_is_console_enabled();

	//override them in case command line parameters are passed
	automounter_early_parse_args(argc,argv, &proc_name, &loglevel, &console_enabled);
	logger_init(proc_name,loglevel,console_enabled);

	//say hello
	logger_log_status("%s",VERSION_INFO_FORMATED_LINE);
}

static void automounter_deinit_device_handler(void)
{
	device_handler_deinit_all();
}

static error_code_t automounter_blacklist_mounted_partitions(void)
{
	FILE *proc_self_mountinfo;
	error_code_t result=RESULT_OK;
	int k;
	char* device;

	device=NULL;

	proc_self_mountinfo=fopen("/proc/self/mountinfo","r");
	if (proc_self_mountinfo!=NULL)
	{
		while(feof(proc_self_mountinfo)==0 && result==RESULT_OK)
		{
			k = fscanf(proc_self_mountinfo,
					"%*s "       /* (1) mount id */
					"%*s "       /* (2) parent id */
					"%*s "       /* (3) major:minor */
					"%*s "       /* (4) root */
					"%*s "       /* (5) mount point */
					"%*s"        /* (6) mount options */
					"%*[^-]"     /* (7) optional fields */
					"- "         /* (8) separator */
					"%*s "       /* (9) file system type */
					"%ms"        /* (10) mount source */
					"%*s"        /* (11) mount options 2 */
					"%*[^\n]",   /* some rubbish at the end */
					&device);

			if (k!=1 && k!=EOF)
			{
				logger_log_error("AUTOMOUNTER - Failed to parse /proc/self/mountinfo.");
				result=RESULT_INVALID;
			}

			if (k==1)
			{
				if (!blacklist_is_system_device(device))
					device_handler_blacklist_mounted_device_all(device);
			}
		}
		if (device!=NULL)
			free(device);

		fclose(proc_self_mountinfo);
	}
	else
	{
		logger_log_error("AUTOMOUNTER - Unable to read out /proc/self/mountinfo. This is needed to properly deal with devices that have already been mounted.");
		result=RESULT_INVALID;
	}

	return result;
}

static bool automounter_unmountall_request(void)
{
	device_t *device;
	device_list_iterator_t itr;

	device=device_list_first_element(&itr);
	while (device!=NULL)
	{
		device_fsm_signal_device_unmount_request(device,NULL,NULL);
		device=device_list_next_element(&itr);
	}

	return mount_is_someone_working();
}

static void automounter_evaluate_mainloop_state_machine(void)
{
	if (automounter_mainloop_state==SHUTDOWN_REQUEST)
	{
		automounter_mainloop_state=SHUTTING_DOWN;
		if (!automounter_unmountall_request())
			//no one requested to be unmounted ...
			automounter_mainloop_state=DONE;
	}
	else if (automounter_mainloop_state==SHUTTING_DOWN)
	{
		if (!mount_is_someone_working())
			//no one is working anymore. Either all partitions have been unmounted
			//properly or some unmounting operations resulted in errors.
			//However, at this point we are unable to do anything more. So if we still
			//have mounted partitions, we are leaving them alone.
			automounter_mainloop_state=DONE;
	}
	else
	{
		//just to satisfy lin(t) (I'm ignoring  _UNSET, DONE, RUNNING )
	}
}

static error_code_t automounter_create_run_dir(void)
{

	if (mkdir(AUTOMOUNTER_RUN_DIR,configuration_get_socket_dir_mode())<0)
	{
		logger_log_error("It seems that the automounter daemon is already running. "
				"If not, remove following directory: " AUTOMOUNTER_RUN_DIR "\n");
		return RESULT_INVALID;
	}

	//Overwrite the umasked-permission by one defined in configuration file.
	(void)chmod(AUTOMOUNTER_RUN_DIR, configuration_get_socket_dir_mode());
	if (chown(AUTOMOUNTER_RUN_DIR,getuid(),configuration_get_socket_group_id()))
	{

	}

	return RESULT_OK;
}

static void automounter_remove_run_dir(void)
{
	rmdir(AUTOMOUNTER_RUN_DIR);
}

//sanity check for the media dir
static error_code_t automounter_check_mountdir(void)
{
	char test_dir[PATH_MAX];
	snprintf(test_dir,PATH_MAX,"%s/test",configuration_get_media_dir());
	logger_log_debug("Try to create a test mount point to check the mount point directory.");

	if (mkdir(test_dir,configuration_get_mountpoint_mode())==-1)
	{
		logger_log_error("Invalid mount point directory. Unable to create a mount point: %s",test_dir);
		return RESULT_INVALID_ARGS;
	}

	if (rmdir(test_dir)==-1)
	{
		logger_log_error("Invalid mount point directory. Unable to remove a mount point: %s",test_dir);
		return RESULT_INVALID_ARGS;
	}

	logger_log_debug("Mount point directory valid.");
	return RESULT_OK;
}
//-------------------------------------------------------------------------------------------------
